<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\License;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Str;

class LicenseAdminController extends Controller
{
    public function index(Request $request)
    {
        $q = trim((string) $request->query('q', ''));
        $status = trim((string) $request->query('status', ''));

        $licenses = License::query()
            ->withCount('activations')
            ->when($q !== '', function ($query) use ($q) {
                $query->where('key', 'like', "%{$q}%");
            })
            ->when($status !== '', function ($query) use ($status) {
                $query->where('status', $status);
            })
            ->latest()
            ->paginate(20)
            ->withQueryString();

        return view('admin.licenses.index', compact('licenses', 'q', 'status'));
    }

    public function create()
    {
        return view('admin.licenses.create');
    }

    public function store(Request $request)
    {
        $data = $request->validate([
            'count' => ['required', 'integer', 'min:1', 'max:500'],
            'max_activations' => ['required', 'integer', 'min:1', 'max:50'],
            'expires_at' => ['nullable', 'date'],
        ]);

        $count = (int) $data['count'];
        $max = (int) $data['max_activations'];
        $expiresAt = $data['expires_at'] ? Carbon::parse($data['expires_at'])->endOfDay() : null;

        $generated = [];

        for ($i = 0; $i < $count; $i++) {
            // Ensure unique key
            do {
                $key = strtoupper(
                    Str::random(4) . '-' . Str::random(4) . '-' . Str::random(4) . '-' . Str::random(4)
                );
            } while (License::where('key', $key)->exists());

            $license = License::create([
                'key' => $key,
                'status' => 'new',
                'max_activations' => $max,
                'expires_at' => $expiresAt,
            ]);

            $generated[] = $license->key;
        }

        return redirect()
            ->route('admin.licenses.index')
            ->with('success', 'Licenses generated: ' . count($generated))
            ->with('generated_keys', $generated);
    }

    public function show(License $license)
    {
        $license->load(['activations' => function ($q) {
            $q->latest();
        }]);

        return view('admin.licenses.show', compact('license'));
    }

    public function resetActivations(License $license)
    {
        $license->activations()->delete();
        $license->update(['status' => 'new']);

        return back()->with('success', 'Activations reset. License can be activated again.');
    }

    public function setStatus(Request $request, License $license)
    {
        $data = $request->validate([
            'status' => ['required', 'in:new,active,revoked,blocked'],
        ]);

        $license->update(['status' => $data['status']]);

        return back()->with('success', 'Status updated.');
    }

    public function delete(License $license)
    {
        $license->delete();
        return redirect()->route('admin.licenses.index')->with('success', 'License deleted.');
    }
}
