<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\License;
use App\Models\LicenseActivation;
use App\Services\Licensing\TokenService;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Validation\ValidationException;

class LicenseApiController extends Controller
{
    public function activate(Request $request, TokenService $tokens)
    {
        $data = $request->validate([
            'license_key'      => ['required', 'string', 'max:80'],
            'fingerprint_hash' => ['required', 'string', 'size:64'],
            'domain'           => ['nullable', 'string', 'max:255'],
            'app_version'      => ['nullable', 'string', 'max:50'],
        ]);

        return DB::transaction(function () use ($data, $request, $tokens) {

            $license = License::where('key', $data['license_key'])->lockForUpdate()->first();
            if (!$license) {
                throw ValidationException::withMessages(['license_key' => 'Invalid license key.']);
            }

            if (in_array($license->status, ['revoked', 'blocked'], true)) {
                throw ValidationException::withMessages(['license_key' => "License is {$license->status}."]);
            }

            if ($license->isExpired()) {
                throw ValidationException::withMessages(['license_key' => 'License is expired.']);
            }

            $count = $license->activations()->count();

            $existingSame = $license->activations()
                ->where('fingerprint_hash', $data['fingerprint_hash'])
                ->first();

            if ($count >= $license->max_activations && !$existingSame) {
                throw ValidationException::withMessages(['license_key' => 'License key already used.']);
            }

            if (!$existingSame) {
                LicenseActivation::create([
                    'license_id'       => $license->id,
                    'fingerprint_hash' => $data['fingerprint_hash'],
                    'domain'           => $data['domain'] ?? null,
                    'ip'               => $request->ip(),
                    'app_version'      => $data['app_version'] ?? null,
                    'activated_at'     => Carbon::now(),
                    'last_check_at'    => Carbon::now(),
                ]);
            } else {
                $existingSame->update([
                    'domain'        => $data['domain'] ?? $existingSame->domain,
                    'ip'            => $request->ip(),
                    'app_version'   => $data['app_version'] ?? $existingSame->app_version,
                    'last_check_at' => Carbon::now(),
                ]);
            }

            if ($license->status === 'new') {
                $license->update(['status' => 'active']);
            }

            $token = $tokens->issue([
                'license_id'       => $license->id,
                'fingerprint_hash' => $data['fingerprint_hash'],
            ], (int) config('licensing.token_ttl_days'));

            return response()->json([
                'success' => true,
                'token'   => $token,
                'status'  => $license->status,
            ]);
        });
    }

    public function validateToken(Request $request, TokenService $tokens)
    {
        $data = $request->validate([
            'token'            => ['required', 'string'],
            'fingerprint_hash' => ['required', 'string', 'size:64'],
            'domain'           => ['nullable', 'string', 'max:255'],
        ]);

        try {
            $payload = $tokens->verify($data['token']);
        } catch (\Throwable $e) {
            throw ValidationException::withMessages(['token' => $e->getMessage()]);
        }

        $license = License::find($payload['license_id'] ?? null);
        if (!$license) {
            throw ValidationException::withMessages(['token' => 'License not found.']);
        }

        if (in_array($license->status, ['revoked', 'blocked'], true)) {
            throw ValidationException::withMessages(['token' => "License is {$license->status}."]);
        }

        if ($license->isExpired()) {
            throw ValidationException::withMessages(['token' => 'License is expired.']);
        }

        $activation = $license->activations()
            ->where('fingerprint_hash', $data['fingerprint_hash'])
            ->first();

        if (!$activation) {
            throw ValidationException::withMessages(['token' => 'Activation not found for this installation.']);
        }

        $activation->update([
            'last_check_at' => Carbon::now(),
            'domain'        => $data['domain'] ?? $activation->domain,
            'ip'            => $request->ip(),
        ]);

        return response()->json([
            'success' => true,
            'status'  => $license->status,
        ]);
    }
}
