<?php

namespace App\Services\Licensing;

use Carbon\Carbon;
use Illuminate\Support\Str;

class TokenService
{
    public function issue(array $payload, int $ttlDays): string
    {
        $payload['iat'] = Carbon::now()->timestamp;
        $payload['exp'] = Carbon::now()->addDays($ttlDays)->timestamp;
        $payload['jti'] = (string) Str::uuid();

        $json = json_encode($payload, JSON_UNESCAPED_SLASHES);
        $b64  = rtrim(strtr(base64_encode($json), '+/', '-_'), '=');

        $sig = hash_hmac('sha256', $b64, $this->secret());

        return $b64 . '.' . $sig;
    }

    public function verify(string $token): array
    {
        $parts = explode('.', $token);
        if (count($parts) !== 2) {
            throw new \RuntimeException('Invalid token format.');
        }

        [$b64, $sig] = $parts;

        $expected = hash_hmac('sha256', $b64, $this->secret());
        if (!hash_equals($expected, $sig)) {
            throw new \RuntimeException('Invalid token signature.');
        }

        $json = base64_decode(strtr($b64, '-_', '+/'));
        $payload = json_decode($json, true);

        if (!is_array($payload)) {
            throw new \RuntimeException('Invalid token payload.');
        }

        if (!isset($payload['exp']) || Carbon::now()->timestamp > (int) $payload['exp']) {
            throw new \RuntimeException('Token expired.');
        }

        return $payload;
    }

    private function secret(): string
    {
        $secret = (string) config('licensing.token_secret');
        if ($secret === '') {
            throw new \RuntimeException('LICENSE_TOKEN_SECRET not configured.');
        }
        return $secret;
    }
}
